package mcfall.raytracer.tests;

import mcfall.math.ColumnVector;
import mcfall.math.IncompatibleMatrixException;
import mcfall.math.Vector;
import junit.framework.TestCase;

public class RowVector extends mcfall.raytracer.tests.TestCase {

	private mcfall.math.RowVector vector;
	private double[] initialValues = {3, 4};
	private mcfall.math.RowVector pointVector;
	public RowVector() {
	}

	@Override
	protected void setUp() throws Exception {
		vector = new mcfall.math.RowVector (initialValues.length, initialValues);
		pointVector = new mcfall.math.RowVector(1);
		pointVector.setValueAt(pointVector.getFirstRowIndex(), pointVector.getFirstColumnIndex(), 5.0);
	}
	public void testRefract() {
		Vector v = new ColumnVector(3,new double[]{1d,-1d,0d});
		Vector n = new ColumnVector(3, new double[]{0,1d,0});
		Vector r = v.refract(n, 1.1d);
		assertAlmostEquals(r.getValueAt(1, 1),0.777817);//thanks wikipedia
		assertAlmostEquals(r.getValueAt(2,1),-0.62849);
		r = v.refract(n, 1.0d);
		assertAlmostEquals(r.getValueAt(1,1),.70710678d);
		assertAlmostEquals(r.getValueAt(2, 1),-.70710678d);//goes straight through
		System.out.println(v.refract(n, 1.0d));
	}
	/*
	 * Test method for 'mcfall.raytracer.RowVector.RowVector(int)'
	 * Ensures that the number of rows is 1, and the number of columns is correct
	 * Also ensures that each value is initialized to 0
	 */
	public void testRowVectorInt() {
		vector = new mcfall.math.RowVector (3);
		assertEquals ("Number of rows in RowVector correct", 1, vector.getNumberOfRows());
		assertEquals ("Number of columns in RowVector correct", 3, vector.getNumberOfColumns());
		for (int column = vector.getFirstColumnIndex(); column <= vector.getLastColumnIndex(); column++) {
			assertEquals ("All initial values set to 0 for RowVector(int) constructor", 0.0, vector.getValueAt(vector.getFirstRowIndex(), column));
		}
	}

	/*
	 * Test method for 'mcfall.raytracer.RowVector.RowVector(int, double[])'
	 */
	public void testRowVectorIntDoubleArray() {
		assertEquals ("Number of rows in RowVector(int, double[]) correct", 1, vector.getNumberOfRows());
		assertEquals ("Number of columns in RowVector(int, double[]) correct", initialValues.length, vector.getNumberOfColumns());
		
		checkInitialValues();
	}

	public void testRowVectorPointPoint () {
		mcfall.math.Point from = new mcfall.math.Point (1.0, 2.0, 3.0);
		mcfall.math.Point to = new mcfall.math.Point (3.0, 5.0, 7.0);
		mcfall.math.RowVector vector = new mcfall.math.RowVector(from, to);
		assertEquals ("X coordinate of row vector constructed using Point,Point version incorrect", 2.0, vector.getValueAt(vector.getFirstColumnIndex()));
		assertEquals ("Y coordinate of row vector constructed using Point,Point version incorrect", 3.0, vector.getValueAt(vector.getFirstColumnIndex()+1));
		assertEquals ("Z coordinate of row vector constructed using Point,Point version incorrect", 4.0, vector.getValueAt(vector.getFirstColumnIndex()+2));
	}
	
	private void checkInitialValues() {
		for (int column = vector.getFirstColumnIndex(); column <= vector.getLastColumnIndex(); column++) {
			assertEquals ("Initial values set correctly for RowVector(int, double[]) constructor", initialValues[column-vector.getFirstColumnIndex()], vector.getValueAt (column));
		}
	}

	/*
	 * Test method for 'mcfall.raytracer.Vector.length()'
	 */
	public void testLength() {
		assertEquals (5.0, vector.length());				
		assertEquals ("Length of single valued row vector correct", 5.0, pointVector.length());
		
		ColumnVector columnVec = new ColumnVector (3);
		columnVec.setValueAt(1, 1.0);
		columnVec.setValueAt(2, 2.0);
		columnVec.setValueAt(3, 3.0);
		assertAlmostEquals(Math.sqrt(14.0), columnVec.length());
	}
	public void testReflect() {
		ColumnVector cv = new ColumnVector(3);
		cv.setValueAt(1, 1.0);
		cv.setValueAt(2, 1.0);
		cv.setValueAt(3, 0.0);
		ColumnVector norm = new ColumnVector(3);
		norm.setValueAt(1, 0.0);
		norm.setValueAt(2, -1.0);
		norm.setValueAt(3, 0.0);
		Vector ref = cv.reflectNormal(norm);
		assertEquals(ref.length(),cv.length());
		assertEquals(ref.getValueAt(1),1.0);
		assertEquals(ref.getValueAt(2),-1.0);
		assertEquals(ref.getValueAt(3),0.0);
		
		
	}
	/*
	 * Test method for 'mcfall.raytracer.Vector.dot(Vector)'
	 */
	public void testDot() {
		//  First test that the correct value is returned when dotting our test vector with itself
		try {
			assertEquals (vector.dot(vector), Math.pow(vector.length(), 2));
		}
		catch (IncompatibleMatrixException invalidException) {
			fail ("Invalid incompatible matrix exception thrown when dotting a vector with a compatible vector");
		}
		
		//  Now test what happens with different signs for the values
		
		//  Finally ensure an exception is thrown when it should be
		try {
			vector.dot(pointVector);
		}
		catch (IncompatibleMatrixException invalidException) {
			return;
		}
		
		fail ("Incompatible matrix exception not thrown when dotting a 2 component row vector with a 1 component row vector");
	}

	/*
	 * Test method for 'mcfall.raytracer.Vector.normalize()'
	 */
	public void testNormalize() {
		//  Ensure the length of the returned vector is 1
		mcfall.math.Vector unit = vector.normalize();
		assertEquals ("Length of normalized vector is 1", 1.0, unit.length());
		//  Ensure that the values of this vector have not been changed
		checkInitialValues();
	}

	/*
	 * Test method for 'mcfall.raytracer.Vector.angleBetweenCosine(Vector)'
	 */
	public void testAngleBetweenCosine() {
		mcfall.math.RowVector x = new mcfall.math.RowVector (2);
		x.setValueAt(1, 1);
		x.setValueAt(2, 0);
		
		mcfall.math.RowVector y = new mcfall.math.RowVector (2);
		y.setValueAt(1, 0);
		y.setValueAt(2, 1);
		
		mcfall.math.RowVector fortyFive = new mcfall.math.RowVector(2);
		fortyFive.setValueAt(1,1);
		fortyFive.setValueAt(2,1);
		
		try {
			double cosineOfAngle = x.angleBetweenCosine(y);
			//  Angle between them is 90 degrees, so cosine should be 0
			assertEquals (0.0, cosineOfAngle);
			
			double correctValue = Math.sqrt(2.0)/2.0;
			
			cosineOfAngle = x.angleBetweenCosine (fortyFive);
			assertAlmostEquals (correctValue, cosineOfAngle);
			
			//  Test this one to get a negative value
			cosineOfAngle = y.angleBetweenCosine (fortyFive);
			assertAlmostEquals (-1*correctValue, -1*cosineOfAngle);
			
		}
		catch (IncompatibleMatrixException invalidIncompatibleMatrix) {
			fail ("Incompatible matrix exception thrown calling angleBetweenCosine when it shouldn't be");
		}
		
	}
	
	public void testSetValueAt () {
		//  Make sure that the correct value is returned by the function
		//  and that the value is actually changed
		
		//  Make sure this works when the first row index is set to something other than 1
		
		//  Ensure that exceptions are thrown when they should be
	}
	
	public void testGetValueAt () {
		//  Ensure the correct value is returned by getValueAt
		
		//  Make sure this works when the first row index is set to something other than 1
		
		//  Ensure that exceptions are thrown when they should be
	}
	
	public void testDuplicate () {
		Vector duplicate = vector.duplicate();
		//  Ensure that the vector returned by duplicate is equal to this vector
		assertTrue (duplicate.equals(vector));
		
		//  And ensure that it isn't the same vector as this one
		assertNotSame("Vector returned from RowVector.duplicate distinct from caller", this, duplicate);
	}

	public void testCrossProduct () {
		double[] i = {1, 0, 0, 0};
		double[] j = {0, 1, 0, 0};
		double[] k = {0, 0, 1, 0};
		
		//  From Hill, page 161
		double[] v1 = {3, 0, 2, 0};
		double[] v2 = {4, 1, 8, 0};
		double[] v1Xv2 = {-2, -16, 3, 0};
		
		double[] w1 = {5,5,-3,0};
		double[] w2 = {0,1,0,0};
		double[] w1Xw2 = {3,0,5,0};
		
		ColumnVector iVec = new ColumnVector(4, i);
		ColumnVector jVec = new ColumnVector(4, j);
		ColumnVector kVec = new ColumnVector(4, k);
		
		ColumnVector v1Vec = new ColumnVector (4, v1);
		ColumnVector v2Vec = new ColumnVector (4, v2);
		ColumnVector v1Xv2Vec = new ColumnVector (4, v1Xv2);
		
		ColumnVector w1Vec = new ColumnVector (4, w1);
		ColumnVector w2Vec = new ColumnVector (4, w2);
		ColumnVector w1Xw2Vec = new ColumnVector (4, w1Xw2);
		
		try {
			Vector result = iVec.cross(jVec);
			assertEquals (result, kVec);
			
			result = jVec.cross(kVec);
			assertEquals (result, iVec);
			
			result = kVec.cross(iVec);
			assertEquals (result, jVec);
			
			result = v1Vec.cross(v2Vec);
			assertEquals (result, v1Xv2Vec);
			
			result = w1Vec.cross(w2Vec);
			assertEquals(result,w1Xw2Vec);
		}
		catch (IncompatibleMatrixException badException) {
			fail ("cross operator threw IncompatibleMatrixException when it shouldn't have");
		}
		
	}
}
